"""server:
co2 xmlrpc server and requesthandler.
"""

import SimpleXMLRPCServer
from threading import Thread

from html_lib import *
import util
#util.print_trace = False # set to true to print traces

__version__ = '0.5.10'

DEFAULT_REQ_PARAMS = ({},)

class Co2Error(Exception):
	"""Co2Error: error class for all things server"""
	pass

class Co2Handler(SimpleXMLRPCServer.SimpleXMLRPCRequestHandler):
	"""Co2Handler: the xmlrpc requesthandler. handles all incoming requests.
	"""
	def _dispatch(self, method_name, params=DEFAULT_REQ_PARAMS):
		"""_dispatch(method_name[, params]):
		dispatch an incoming xmlrpc call.
		if the second argument is missing, it is set to a tuple containing an empty
		dictionary.
		the key 'client_address' is set to the address of the request's originating
		host, the value is a tuple (<host>, <port>). the host part is used later on,
		e.g. to verify requests.
		method_name: string, name of the method to be called;
		params: tuple, optional parameters to be passed on to the method
		(as first element of the tuple. why tuple?
		SimpleXMLRPCServer.SimpleXMLRPCRequestHandler uses apply() which expects a
		tuple as argument parameter).
		returns whatever the called method returns.
		"""
		#print 'HANDLER dispatch RAW:', method_name, params
		if len(params) == 0:
			params = DEFAULT_REQ_PARAMS
		params[0]['client_address'] = self.client_address
		#print 'HANDLER dispatch COOKED:', method_name, params
		#print 'HANDLER dispatch CLIENT:', method_name, params
		# get method from the registry (the server's registered instance)
		# and call it with params
		return apply(self.server.instance.methods[method_name], params)
		#return self.server.instance.methods[method_name](params[0])

	def _dispatch_html(self, method_name, param_dict={}):
		"""_dispatch_html(method_name[, param_dict]):
		dispatch an html GET request. basically the same as _dispatch, but restricted
		to methods whose names begin with 'web.'.
		method_name: string, name of the method to be called;
		param_dict: optional dictionary, to be passed on to the method as argument.
		returns whatever the called method returns.
		"""
		if not method_name.startswith('web.'):
			raise  Co2Error('method "%s" is not callable through the webinterface.' % method_name)
		#print 'XXX', method_name, param_dict
		#return self._dispatch(method_name, param_dict)
		return self.server.instance.methods[method_name](param_dict)

	def do_GET(self):
		"""do_GET():
		called by the underlying http server on a GET request.
		returns html formatted data since we assume the request comes from a
		web browser. see html_factory.parse_qs()
		"""
		msg = ''
		stuff = ''
		page_type = 'index'
		rest = None
		response_code = 200
		try:
			self.path.index('?')
			self.path, rest = self.path.split('?', 1)
		except ValueError:
			self.path = self.path.split('?', 1)[0]
		if self.path == '/satellites':
			page_type = 'satellites'
			stuff = 'currently registered satellites:<p />'
			try:
				satellites = self._dispatch_html('web.get_satellites')
				if len(satellites) == 0:
					stuff = '%snone' % stuff
				for satellite in satellites:
					sat_props = self._dispatch_html('web.get_satellite_props',
																					{'host_id':satellite})
					stuff = '%s%s' % (stuff, html_factory.dict_to_table(sat_props, 'nick'))
			except Exception, err_msg:
##				print '#' * 64
##				print 'err_msg', err_msg
##				print '#' * 64
				msg = '<pre>%s</pre>' % util.trace()
				response_code = 500
		elif self.path == '/methods':
			page_type = 'methods'
		else:
			server_props = self._dispatch_html('web.get_properties')
			stuff = '%s%s' % (stuff, html_factory.dict_to_table(server_props))

		self.send_response(response_code)
		self.send_header('Content-type', 'text/html')
		self.end_headers()
		self.wfile.write(html_factory.make_page(page_type, msg, stuff))

class Co2Server(SimpleXMLRPCServer.SimpleXMLRPCServer):
	"""Co2Server: co2 subclass of SimpleXMLRPCServer.SimpleXMLRPCServer"""
	allow_reuse_address = 1
	
	def __init__(self, vars):
		"""__init__(vars):
		first calls the superclass's init, then does some co2 stuff.
		vars: dictionary, server specific variables.
		"""
		SimpleXMLRPCServer.SimpleXMLRPCServer.__init__(self, vars['addr'], Co2Handler)
		self.vars = vars
		self.log = vars['log_method']
		self.debug = vars['debug_method']
		if self.vars['threaded']:
			self.process_request = self.process_request_threaded
		self.exit_hooks = []
		html_factory.registry_type = vars['registry']
		html_factory.address = self.server_address
		self.system_properties = {}
		self._set_system_properties()

	def _set_system_properties(self):
		from sys import version_info
		major, minor, micro, releaselevel, serial = version_info
		self.system_properties['sw_platform'] = 'Python %d.%d.%d %s' % (major,
																																		minor,
																																		micro,
																																		releaselevel)
		del version_info

		from os import popen
		uname_dict = {'operating_system':'o',
									'hardware':'m',
									'processor':'p',
									'hw_platform':'i',
									'kernel':'r'}
		for key in uname_dict:
			try:
				fd = popen('uname -%s' % uname_dict[key], 'r')
				self.system_properties[key] = fd.readline()
				if fd is not None:
					fd.close()
			except:
				self.system_properties[key] = 'unknown'
		del popen

	def register_instance(self, instance):
		SimpleXMLRPCServer.SimpleXMLRPCServer.register_instance(self, instance)
		html_factory.method_table = html_factory.make_method_table(self.instance.methods)
		self.instance.system_properties.update(self.system_properties)

	def listen(self):
		"""listen():
		start the handle-request loop.
		"""
		self.debug(self, 'listening...', 2)
		self.serve_forever()
		self.debug(self, 'Co2Server stopped listening.', 2)

	def process_request_thread(self, request, client_address):
		"""process_request_thread(request, client_address):
		Same as in BaseHTTPServer but as a thread and, in addition,
		exception handling is done here.
		"""
		try:
			self.finish_request(request, client_address)
			self.close_request(request)
		except:
			#util.trace()
			self.handle_error(request, client_address)
			self.close_request(request)

	def process_request_threaded(self, request, client_address):
		"""process_request_threaded(request, client_address):
		start a new thread to process the request.
		"""
		t = Thread(target = self.process_request_thread,
							 args = (request, client_address))
		t.start()

	def register_exit_hook(self, method):
		"""register_exit_hook(method):
		register a method to be called on exit.
		method: method object, method to be registered as exit hook.
		"""
		self.exit_hooks.append(method)

	def clean_up(self):
		"""clean_up():
		clean up on server exit. call registered exit hooks, wait a little,
		then the server exits, because this is the last thing done.
		"""
		self.debug(self, 'Co2Server: cleaning up...', 2)
		for hook in self.exit_hooks:
			self.debug(self,
								 'Co2Server: calling exit_hook "%s.%s"' % (hook.im_class,
																													 hook.__name__),
								 0)
			hook()
		# wait a little, so we don't exit before our children
		# ugly, but else a funny exception is raised, with only a partial trace
		from time import sleep
		self.debug(self, '...waiting...', 0)
		sleep(2) # TODO: _wirklich_ auf andere threads warten?
		del sleep
		self.debug(self, 'EXIT', 0)

	def __str__(self):
		return 'Co2Server [%s:%d]' % self.server_address
	__repr__ = __str__

