"""satellite
representation of a co2 satellite, used by relay_registry.Co2RelayRegistry
"""

from threading import Thread
from xmlrpclib import DateTime
from time import time

import client
#import util

class SatelliteError(Exception): pass

class Satellite:
	"""Satellite:
	represents a co2 satellite host. has it's own proxy to call methods on the
	satellite, and holds various statistical data as instance variables.
	"""
	__props__ = ('host_id', 'nick', 'addr', 'ctime', 'recv_time', 'send_time',
							 'recv_count', 'recv_count', 'ping_current', 'ping_min',
							 'ping_max', 'ping_avg', 'ping_fails', 'ping_count', 'plugins')

	def __init__(self, host_id, addr, debug_method, nick='', plugins=[]):
		"""__init__(host_id, addr, debug_method):
		an internal representation of a co2 satellite.
		host_id: string, the satellite's ID;
		addr: tuple (<host>, <port>), the address of the satellite;
		debug_method: method object, the global debug method.
		"""
		self.host_id = host_id
		if nick == '': nick = host_id
		self.nick = nick
		self.addr = addr

		self.proxy = client.Co2Proxy(addr)

		self.ctime = time()

		self.recv_time = self.ctime
		self.recv_count = 0

		self.send_time = self.ctime
		self.send_count = 0

		self.ping_current = -1
		self.ping_min = -1
		self.ping_max = -1
		self.ping_avg = -1
		self.ping_fails = 0
		self.ping_count = 0

		self.debug = debug_method

		self.plugins = plugins

	def set_props(self, props_dict):
		for key in self.__props__:
			try:
				setattr(self, key, props_dict[key])
			except KeyError, msg:
				self.debug(self, 'failed to set property "%s" [%s]' % (key, msg), 2)

	def get_props(self):
		return {'host_id':self.host_id,
						'nick':self.nick,
						'addr':self.addr,
						'plugins':self.plugins,
						'ctime':self.ctime,
						'recv_time':self.recv_time,
						'recv_count':self.recv_count,
						'send_count':self.send_count,
						'send_time':self.send_time,
						'ping_fails':self.ping_fails,
						'ping_current':self.ping_current,
						'ping_min':self.ping_min,
						'ping_max':self.ping_max,
						'ping_avg':self.ping_avg,
						'ping_count':self.ping_count}

	#props = property(get_props, set_props, None, 'satellite properties')

	def touch_recv_time(self):
		"""touch_recv_time():
		updates self.recv_time, called whenever a notify-call goes out
		through self.proxy, and increments self.recv_count by one.
		"""
		self.recv_time = time()
		self.recv_count += 1

	def touch_send_time(self):
		"""touch_send_time():
		updates self.send_time, called whenever a relay-call comes in
		from the satellite, and increments self.send_count by one.
		"""
		self.send_time = time()
		self.send_count += 1

	def set_ping_current(self, dt):
		self.ping_count += 1
		self.ping_current = dt
		if self.ping_count == 1:
			self.ping_avg = self.ping_current
			self.ping_min = self.ping_current
			self.ping_max = self.ping_current
		else:
			self.ping_avg += (self.ping_current - self.ping_avg) / self.ping_count
			self.ping_min = min(self.ping_min, self.ping_current)
			self.ping_max = max(self.ping_max, self.ping_current)
		self.ping_fails = 0

	def increment_ping_fails(self):
		self.ping_fails += 1
		return self.ping_fails

	def notify(self, param_dict):
		"""notify(param_dict):
		calls the satellites notify method.
		param_dict: dictionary (data to be relayed).
		"""
		if self.ping_fails == 0:
			self.debug(self, 'notify [%s]' % str(param_dict), 0)
			Thread(target = self.notify_thread, args = (param_dict,)).start()
			self.touch_recv_time()
		else:
			self.debug(self,
								 'not notified, failed pings: %s' % self.ping_fails,
								 2)

	def notify_thread(self, param_dict):
		"""notify_thread(param_dict):
		notify calls are threaded, so a lame satellite or a bad connection
		don't hurt the relay. called by notify(param_dict).
		param_dict: dictionary (data to be relayed, simply passed on).
		"""
		#print 'Satellite notify thread:', host_id, method_name, data
		try:
			#self.proxy.proxy.notify(param_dict)
			getattr(self.proxy.proxy, 'notify')(param_dict)
		except Exception, msg:
			#util.trace()
			self.debug(self, 'error in notify thread [%s]' % str(msg), 3)

	def get_status(self):
		"""get_status()
		returns a dictionary with various data about this satellite.
		format:
		'host_id':string, the satellites ID;
		'addr': tuple, the satellite's address;
		'plugins': list, loaded plugins;
		'ctime': iso8601 DateTime, timestamp of registration (creation);
		'recv_time': iso8601 DateTime, timestamp of most recent receive;
		'recv_count': int, number of messages received;
		'send_time': iso8601 DateTime, timestamp of most recent send;
		'send_count': int, number of messages sent;
		'ping_fails': int, number of currently unanswered pings, reset to 0 on
		successful ping;
		'ping_current': float, most recent ping delay;
		'ping_min': float, minimum ping time;
		'ping_max': float, maximum ping time;
		'ping_avg': float, average ping time;
		'ping_count': int, number of successfull pings.
		"""
		return {'host_id':self.host_id,
						'nick':self.nick,
						'addr':self.addr,
						'plugins':self.plugins or [],
						'ctime':DateTime(self.ctime),
						'recv_time':DateTime(self.recv_time),
						'recv_count':self.recv_count,
						'send_count':self.send_count,
						'send_time':DateTime(self.send_time),
						'ping_fails':self.ping_fails,
						'ping_current':self.ping_current,
						'ping_min':self.ping_min,
						'ping_max':self.ping_max,
						'ping_avg':self.ping_avg,
						'ping_count':self.ping_count}

##	def get_dump_data(self):
##		"""get_dump_data():
##		returns a dictionary (host_id, addr) to be dumped to a file
##		when the relay exits.
##		see: Co2RelayRegistry._serialize_sat_data
##		"""
##		return {'host_id':self.host_id, 'addr':self.addr, 'nick':self.nick}

	def __str__(self):
		return '%s [%s]' % (self.__class__.__name__, self.host_id)
	__repr__ = __str__

if __name__ == '__main__':
	def debug(caller, msg, level=0):
		print '%s: %s' % (caller, msg)
	s = Satellite('ID', ('', 0.815), debug)
	s.set_last_ping(8.9)
	print s.get_status()
