"""relay
the co2 registry with relay functionality
"""

from threading import Thread

import registry, client, util
from satellite import *
from checker import *

class Co2RelayRegistry(registry.Co2Registry):
	"""Co2RelayRegistry: provides the relay specific methods.
	"""
	def __init__(self, vars):
		"""__init__(vars):
		first calls Co2Registry's __init__, then does some initial stuff for
		its own (load eventual satellite dump file, init and kick off the
		checker.
		vars: dictionary, registry specific variables.
		"""
		registry.Co2Registry.__init__(self, vars)
		self.system_properties['max_sats'] = vars['max_sats']
		self.satdump_filename = 'saved_sats.xml'
		self.satellites = {}
		self.debug(self, 'looking for dumped satellite data.', 1)
		sat_data = util.read_sat_data(self.vars['co2_home'], self.satdump_filename)
		for sd in sat_data:
			self.satellites[sd['host_id']] = Satellite(sd['host_id'],
																								 sd['addr'],
																								 self.debug)
			self.satellites[sd['host_id']].set_props(sd)
			self.debug(self,
								 'found and loaded "%s"' % self.satellites[sd['host_id']],
								 1)
		self.checker = Checker()
		self.checker.init(self.satellites, self.debug)
		self.checker.start()
		self.limiter.init(self._relay_real, self.debug)
		self.limiter.start()

	def _exit_hook(self):
		"""_exit_hook():
		called on server exit. triggers cleaning up.
		"""
		# first do relay specific cleaning up
		if len(self.satellites) > 0:
			try:
				self._serialize_sat_data()
			except Exception, msg:
				self.debug(self,
									 'WARNING: failed to dump satellite sata! [%s]' % msg,
									 2)
		else:
			# there are no satellites registered, so remove a dump that might be around
			try:
				if util.remove_file(self.vars['co2_home'], self.satdump_filename):
					self.debug(self, 'removed satellite dump file.', 0)
			except Exception, msg:
				self.debug(self,
									 'could not remove file "%s"' % self.satdump_filename,
									 2)
		self.debug(self,
							 'satellite data dumped: %d' % len(self.satellites),
							 1)
		self.checker.halt()
		# then call registry's exit hook
		registry.Co2Registry._exit_hook(self)
		#self.limiter.halt()

	def _serialize_sat_data(self):
		"""_serialize_sat_data():
		dump an xml file to disc containing all data needed for the relay to
		re-register the satellites automatically on restart, so that satellites
		don't have to register again themselves.
		"""
		if not util.can_marshal:
			self.debug(self,
								 'no marshalling library available. satellite data not saved.',
								 2)
			return
		sat_data = []
		for sat_id in self.satellites.keys():
			sat_data.append(self.satellites[sat_id].get_props())
		try:
			util.dump_sat_data(sat_data,
												 self.vars['co2_home'],
												 self.satdump_filename)
		except Exception, msg:
			self.debug(self, 'could not dump satellite data [%s]' % msg, 2)

	def _relay_real(self, param_dict):
		self.debug(self, 'relay: %s' % param_dict, 0)
		self._verify_registration(param_dict['host_id'])
		# TODO:
		# record frequency of calls per sat
		# -> pause relaying for sats that are too talkative
		self.satellites[param_dict['host_id']].touch_send_time()
		for sat_id in self.satellites.keys():
			if param_dict['host_id'] != sat_id:
				# relay to (all other) satellites
				try:
					self.satellites[sat_id].notify(param_dict)
				except Exception, msg:
					self.debug(self,
										 'relay: failed to notify %s [%s]' % (sat_id,
																													str(msg)),
										 2)
		return 'OK'

	########################################
	# xmlrpc methods

	def x_relay_register(self, param_dict):
		"""x_relay_register(param_dict):
		satellites that want to use relaying must call this method first.
		param_dict: satellite's addr (host, port). subsequently used for sending
		to the satellite.
		returns the id the satellite has to use for further communication
		with the relay.
		"""
		# we take the "real" IP as provided by the server's handler and the port
		# as supplied by the client to get a reliable return address.
		if len(self.satellites) >= self.vars['max_sats']:
			raise registry.Co2RegistryError('max. # of sats (%d) reached, sorry...' % self.vars['max_sats'])
		client_address = (param_dict['client_address'][0],
											param_dict['port'])
		self.debug(self, 'register request for "%s:%d"' % client_address, 1)
		host_id = '%s:%d' % client_address
		self.satellites[host_id] = Satellite(host_id,
																				 client_address,
																				 self.debug,
																				 param_dict['nick'],
																				 param_dict['plugins'])
		return host_id

	def x_relay_deregister(self, param_dict):
		"""x_relay_deregister(param_dict):
		revoke a satellite's registration.
		should be called by satellites before they shut down. the calling client's
		address must match the supplied host id.
		param_dict: {'host_id':xxx}.
		returns 'OK', or raises Exception on failure.
		"""
		host_id = param_dict['host_id']
		self._verify_registration(host_id)
		sat = self.satellites[host_id]
		# a little paranoia: make sure satellites only deregister themselves
		self._match_id(param_dict['client_address'], sat)
		del self.satellites[host_id]
		#del self.checker.satellites[host_id]
		self.debug(self, 'deregistered %s' % host_id, 1)
		return 'OK'

##	def x_relay_relay(self, param_dict):
##		"""x_relay_relay(param_dict):
##		incoming call with data to relay to our satellites.
##		param_dict: data to be relayed, must contain a key 'host_id' (value:
##		the id provided by the relay on registration, and a key 'data' (value:
##		whatever data the satellite wants to send).
##		returns 'OK', or raises Co2RegistryError on failure.
##		"""
##		self.debug(self, 'relay: %s' % param_dict)
##		self._verify_registration(param_dict['host_id'])
##		# TODO:
##		# record frequency of calls per sat
##		# -> pause relaying for sats that are too talkative
##		self.satellites[param_dict['host_id']].touch_send_time()
##		for sat_id in self.satellites.keys():
##			if param_dict['host_id'] != sat_id:
##				# relay to (all other) satellites
##				try:
##					self.satellites[sat_id].notify(param_dict)
##				except Exception, msg:
##					self.debug(self, 'relay: failed to notify %s [%s]' % (sat_id,
##																																str(msg)))
##		return 'OK'

	def x_relay_relay(self, param_dict):
		"""x_relay_relay(param_dict):
		incoming call with data to relay to our satellites.
		param_dict: data to be relayed, must contain a key 'host_id' (value:
		the id provided by the relay on registration, and a key 'data' (value:
		whatever data the satellite wants to send).
		returns 'OK', or raises Co2RegistryError on failure.
		"""
		self.limiter.enqueue(param_dict)
		return 'OK'

	def x_relay_get_max_satellites(self, param_dict={}):
		"""x_relay_get_max_satellites():
		get the maximum number of satellites that are allowed to be registered.
		returns: int, satellite limit.
		"""
		return self.vars['max_sats']

	def x_relay_get_satellites(self, param_dict={}):
		"""x_relay_get_satellites([param_dict]):
		returns a list if IDs, each of which can be passed to get_satellite
		(as {'host_id':<ID>}).
		"""
		return self.satellites.keys()
	x_co2_get_satellites = x_web_get_satellites = x_relay_get_satellites

	def x_relay_get_satellite_props(self, param_dict):
		"""x_relay_get_satellite(param_dict):
		get status / statistical information from a satellite.
		param_dict: {'host_id':nnn} specifies a satellite id.
		returns a dictionary:
		addr: list ([<ip>, <port>]);
		host_id: string (the satellite host's id);
		ctime: DateTime (time of registration with the relay);
		ping_current: float (last ping time in msecs, -1 if not pinged);
		ping_fails: int (# of failed ping requests by the relay);
		send_time: DateTime (most recent message originating from this satellite);
		recv_time: DateTime (most recent message received by this satellite).
		"""
		try:
			return self.satellites[param_dict['host_id']].get_status()
		except Exception, msg:
			raise registry.Co2RegistryError('failed to get status from %s [%s]' % param_dict['host_id'], msg)
	x_co2_get_satellite_props = x_web_get_satellite_props = x_relay_get_satellite_props
	x_co2_get_satellite = x_relay_get_satellite_props


	# /xmlrpc methods
	########################################
	# helpers

	def _verify_registration(self, host_id):
		"""_verify_registration(host_id):
		host_id: co2 host id as provided by the relay.
		raises a Co2RegistryError if no sat with the specified id is registered.
		"""
		if not host_id in self.satellites.keys():
			raise registry.Co2RegistryError('%s is not registered.' % host_id)

	def _match_id(self, address, satellite):
		"""_match_id(address, satellite):
		address: tuple (host, port);
		satellite: co2 Satellite instance.
		raises a Co2RegistryError if the hosts of address and satellite don't match.
		"""
		# a little paranoia: used to make sure sat_id (provided by sat)
		# and client_address (passed on by the server's requesthandler) match
		if satellite.addr[0] != address[0]:
			raise registry.Co2RegistryError('id mismatch [%s:%d - %s]' % (address,
																																		satellite.host_id))

##	def _make_host_id(self, addr):
##		"""_make_host_id(addr) -> host id
##		create a unique host id. the supplied address (ip:port) is prefixed by the
##		current timestamp (could be a little more creative...)
##		returns a unique host id.
##		"""
##		return '%s@%s:%d' % (str(int(time())), addr[0], addr[1])

	# /helpers
	########################################

