"""registry:
the core of co2's xmlrpc functionality.
"""

import inspect, os
from sys import path as sys_path
from sys import exit as sys_exit
from threading import Thread
from xmlrpclib import DateTime
from time import time

import client, loader
from limiter import Limiter

class Co2RegistryError(Exception): pass

class Co2Method:
	"""Co2Method:
	registry-internal representation of a registered (i.e. callable)
	xmlrpc method.
	"""
	def __init__(self, method):
		self.method = method
		self.doc = method.__doc__ or 'no help available'
		#TODO: inspect -> sig
		
		#self.signature = 'no signature available'
		self.signature = format_argspec(inspect.getargspec(self.method))
		#und wenn wir lustig sind, implementieren wir ein get_code ;]

	def __call__(self, param_dict):
		"""__call__(param_dict):
		call the specified method.
		param_dict: dictionary (args to the method, passed through).
		returns whatever the called method returns.
		"""
		return self.method(param_dict)

	def __str__(self):
		return 'Co2XMLRPCMethod "%s"' % self.method.__name__

def format_argspec(argspec):
	"""format_argspec(argspec):
	create a method signature, will be returned by calls to
	system.methodSignature(<method_name>) and aliases thereof.
	argspec: tuple, as returned by inspect.getargspec(method).
	returns: dictionary, key:value pairs are:
	'args':arguments with no default values (list);
	'args_d':arguments with default values (dictionary {'arg':<default value>};
	'args_v':variable arguments (list);
	'args_kw':keyword arguments (list).
	(the latter two are python specific and of no relevance to xmlrpc.)
	"""
	f_args = []
	f_args_w_defaults = {}
	args = argspec[0]
	args.reverse()
	varargs = argspec[1]
	if varargs is None: varargs = []
	varkw = argspec[2]
	if varkw is None: varkw = []
	defaults = argspec[3]
	if defaults is None:
		defaults = []
	else:
		defaults = list(defaults)
		defaults.reverse()
	#print args
	for i in range(len(args)):
		#for i in range(-1, (len(args) + 1) * -1, -1):
		if args[i] == 'self': continue
		try:
			f_args_w_defaults[args[i]] = defaults[i]
		except IndexError:
			f_args.append(args[i])
	return {'args':f_args,
					'args_d':f_args_w_defaults,
					'args_v':varargs,
					'args_kw':varkw}

##class Co2XMLRPCMethod(Co2Method):
##	pass
##class Co2WIMethod(Co2Method):
##	pass

def make_reg_name(name):
	"""make_reg_name(name):
	create a name under which a method will be registered for access
	through xmlrpc.
	name: string, the name of a co2 method to be registered.
	naming conventions:
	prefixes determine whether or not and if under
	what name methods are registered (ie. XMLRPC-callable)
	 methname      XMLRPCname     XMLRPC call
	 x_top_<name>  <name>         proxy.<name>()
	 x_co2_<name>  co2.<name>     proxy.co2.<name>()
	 x_sys_<name>  system.<name>  proxy.system.<name>()
	 (the last one is for userland (xmlrpc.com) compatibility)
	"""
	if name.startswith('x_sys_'): name = 'system.%s' % name[6:]
	elif name.startswith('x_top_'): name = name[6:]
	elif name.startswith('x_co2_'): name = 'co2.%s' % name[6:]
	elif name.startswith('x_relay_'): name = 'relay.%s' % name[8:]
	elif name.startswith('x_web_'): name = 'web.%s' % name[6:]
	# we can assume the rest simply has 'x_' prepended and we
	# remove that for use in plugins where the methodname is
	# appended to package- and module-names, like: pkg.mod.meth
	# don't use in relay or satellite (use x_top_)
	else: name = name[2:]
	return name

class Co2Registry:
	"""Co2Registry:
	holds all xmlrpc methods, registers system methods, initializes the
	plugin loader, and handles all incoming calls.
	"""
	def __init__(self, vars):
		"""__init__(vars):
		do all init stuff (set variables, load methods and start the loader).
		"""
		self.vars = vars
		self.debug = vars['debug_method']
		self.log = vars['log_method']
		self.receivers = [] # receive methods of plugins
		if self.vars['relay_addr']:
			# there's a relay defined, so we need a proxy to talk to it
			# TODO: also use this proxy for relay registry?
			self.relay_proxy = client.Co2Proxy(self.vars['relay_addr'],
																				 self.vars['socket_timeout'])
		self.host_id = 'unregistered@%s:%d' % self.vars['addr']
		self.system_properties = {'co2_version':self.vars['co2_version'],
															'co2_author':self.vars['co2_author'],
															'maintainer':self.vars['maintainer'],
															'maintainer_email':self.vars['maintainer_email'],
															'plugins':self.vars['plugin_list'] or [],
															'start_time':time()}
		plugin_vars = {}
		plugin_vars['relay_method'] = self._relay
		plugin_vars['debug_method'] = vars['debug_method']
		plugin_vars['log_method'] = vars['log_method']
		self.debug(self,
							 'relay_method = %s' % plugin_vars['relay_method'].__name__,
							 0)
		self.methods = {}
		self._load_system_methods()
		self.plugin_exit_hooks = []
		loader_vars = {'base_dir':self.vars['plugin_path'],
									 'plugin_list':self.vars['plugin_list'],
									 'debug_method':self.vars['debug_method'],
									 'log_method':self.vars['log_method']}
		self._load_plugins(loader_vars, plugin_vars)
		self.limiter = Limiter()

	def _relay(self, data_dict):
		# desisschiach, pfoa!
		# call_relay sachen jetzt in sat_reg, wo sie eigentlich hingehoeren,
		# aber die sat plugins, die die _relay methode brauchen, werden hier
		# oben initialisiert...
		self.debug(self, 'no relay method available', 2)

	def _load_system_methods(self):
		"""_load_system_methods()
		register system methods.
		system methods provide meta-information, or utility functions
		(like list all available [callable] methods etc.)
		"""
		for name, method in inspect.getmembers(self, is_co2_method):
			self._register_method(make_reg_name(name), method)
		self.debug(self, '%d builtin methods' % len(self.methods), 1)

	def _load_plugins(self, loader_vars, plugin_vars):
		"""_load_plugins(loader_vars, plugin_vars):
		kick off the plugin loader.
		loader_vars: dictionary, relevant vars for the loader;
		plugin_vars: dictionary, relevant vars for the plugins.
		"""
		l = loader.Loader(loader_vars, plugin_vars)
		self.receivers = l.load_all(self.plugin_exit_hooks)
		self.debug(self, 'receivers:', 0)
		for receiver in self.receivers:
			self.debug('\t', '%s.%s.%s' % (receiver.im_class.__module__,
																		 receiver.im_class.__name__,
																		 receiver.im_func.__name__), 0)
		#self.debug(self, 'receivers:\n%s' % str(self.receivers))

	def _register_method(self, name, method):
		"""_register_method(name, method):
		register a method to be callable through xmlrpc.
		name: string, the name under which the method will be callable;
		method: the method object itself.
		"""
		self.methods[name] = Co2Method(method)
		self.debug(self, 'registered %s as %s' % (method.__name__, name), 1)

	def _exit_hook(self):
		"""_exit_hook():
		called by the server on exit, in turn calls plugins' exit hooks,
		to clean up before exiting.
		"""
		self.debug(self, 'exit_hook called.', 1)
		for hook in self.plugin_exit_hooks:
			self.debug(self, 'calling %s...' % hook, 1)
			hook()
		self.limiter.halt()

	def _notify_thread(self, recv_method, param_dict):
		'''_notify_thread(recv_method, param_dict):
		'''
		recv_method(param_dict)

	def _notify(self, recv_method, param_dict):
		Thread(target = self._notify_thread, args = (recv_method, param_dict)).start()

##	def _relay(self, data_dict):
##		"""relay(data_dict):
##		convenience method on top of _call_relay(). equivalent to
##		_call_relay('relay.relay', data_dict).
##		data_dict: dictionary, data to be relayed.
##		"""
##		#return self._call_relay('relay.relay', data_dict)
##		if not data_dict.has_key('data'):
##			raise Co2RegistryError('no key "data" found. nothing to send...')
##		return self._call_relay('relay.relay', data_dict)

##	def _call_relay(self, method_name, param_dict={}):
##		"""_call_relay(method_name[, param_dict]):
##		start a thread that calls method_name with argument param_dict on the relay .
##		method_name: name of the method to be called;
##		param_dict: optional parameter dictionary, default: {}.
##		"""
##		Thread(target = self._call_relay_thread, args = (method_name, param_dict)).start()
##		return 'OK'

##	def _call_relay_thread(self, method_name, param_dict):
##		"""_call_relay_thread(method_name, param_dict):
##		outgoing call to the relay. exceptions are cought and not passed on to the
##		calling host, except when method_name is 'relay.register' a Co2RegistryError
##		is raised, so that satellites can react, probably exit.
##		method_name: name of the relay's method to be called;
##		param_dict: dictionary, argument to the method.
##		"""
##		#print 'REGISTRY _call_relay RAW:', method_name, param_dict
##		param_dict['host_id'] = self.host_id# or 'unregistered@%s:%d' % self.vars['addr']
##		#print 'REGISTRY _call_relay +ID:', method_name, param_dict
##		try:
##			self.debug(self, 'calling relay ["%s%s"]' % (method_name, str(param_dict)))
##			return getattr(self.relay_proxy.proxy, method_name)(param_dict)
##		except Exception, msg:
##			param_dict.pop('host_id')
##			self.debug(self,
##								 'WARNING: call "%s(%s)" failed: %s' % (method_name,
##																												str(param_dict),
##																												msg))
##			if method_name == 'relay.register':
##				raise Co2RegistryError('ERROR: unable to register. relay unreachable?')

	def __str__(self):
		return self.__class__.__name__



	####################################################
	# co2 XMLRPC system methods below
	#
	# prefixes determine under what name methods will be
	# registered/accessible, see make_reg_name()

	############
	# top level

	def x_top_ping(self, param_dict={}):
		"""x_top_ping():
		answer a ping request.
		returns 'pong'.
		alias: x_web_ping.
		callable as ping, web.ping.
		"""
##		#to test socket timeout
##		from time import sleep
##		sleep(5)
		return 'pong'
	x_web_ping = x_top_ping # registered separately -> alias

	def x_top_test(self, param_dict={}):
		"""x_top_test([param_dict]):
		just a test method...
		param_dict: optional argument (dictionary).
		returns the argument (param_dict).
		alias: x_web_test.
		callable as test, web.test.
		"""
		return 'test OK: %s received %s' % (self.__class__, str(param_dict))
	x_web_test = x_top_test

	def x_top_call_method(self, param_dict):
		"""x_top_call_method(param_dict):
		'anonymized' call method.
		param_dict: dictionary with members:
		most importantly: 'method_name':'method_name' plus all args that method
		expects (signature returned by x_co2_get_method_help).
		returns whatever the called method returns.
		alias: x_web_call_method.
		callable as call_method, web.call_method.
		"""
		method_name = param_dict.pop('method_name')
		return self.methods[method_name](param_dict)
	x_web_call_method = x_top_call_method

	def x_top_notify(self, param_dict):
		"""x_top_notify(param_dict):
		to relay data to satellites, the relay calls this method which in turn
		calls all receive methods registered by plugins and passes on param_dict.
		param_dict: dictionary, data to be sent to plugins.
		returns 'OK' or raises an exception on failure.
		callable as notify.
		"""
		self.debug(self, 'notify [%s]' % param_dict, 0)
		for recv in self.receivers:
			try:
				self._notify(recv, param_dict)
			except Exception, msg:
				self.debug(self, 'receiver error [%s]' % str(msg), 3)
				return False
		return 'OK'

	############
	# co2 level

	def x_co2_get_method_names(self, param_dict={}):
		# TODO: param_dict = prefix -> filter (?)
		"""x_co2_get_method_names():
		returns a list of available methods.
		aliases: x_sys_listMethods, x_web_get_method_names.
		callable as co2.co2_get_method_names, sys.listMethods, web.get_method_names.
		"""
		return self.methods.keys()
	x_sys_listMethods = x_co2_get_method_names
	x_web_get_method_names = x_co2_get_method_names

	def x_co2_get_method_help(self, param_dict):
		"""x_co2_get_method_help(param_dict):
		param_dict: dictionary, mandatory key: 'method_name', value:
		name of a method (from list returned by x_co2_get_method_names.
		returns a string with a brief description of what the method does.
		aliases: x_sys_methodHelp, x_web_get_method_help.
		callable as co2.get_method_help, sys.methodHelp, web.get_method_help.
		"""
		return self.methods[param_dict['method_name']].doc
	x_sys_methodHelp = x_co2_get_method_help
	x_web_get_method_help = x_co2_get_method_help

	def x_co2_get_method_signature(self, param_dict):
		"""x_co2_get_method_signature(param_dict):
		returns a discription of how to use the specified method
		(format: a string for now, might well change).
		param_dict: dictionary with mandatory key: 'method_name', value:
		(callable) name of a method registered with this registry instance.
		aliases: x_sys_methodSignature, x_web_get_method_signature.
		callable as co2.get_method_signature, sys.methodSignature,
		web.get_method_signature.
		"""
		return self.methods[param_dict['method_name']].signature
	x_sys_methodSignature = x_co2_get_method_signature
	x_web_get_method_signature = x_co2_get_method_signature

	############
	# sys level
	def x_sys_get_properties(self, param_dict={}):
		"""x_sys_get_properties():
		get system properties.
		returns a dictionary [key: value-type (description)]:
		co2_author: string (server author);
		kernel: string (host kernel name);
		uptime: int (server uptime in seconds);
		operating_system: string (host operating system);
		maintainer: string (server maintainer);
		start_time: DateTime (server start time);
		sw_platform: string (co2 software platform);
		hw_platform: string (host hardware platform);
		co2_version: string (co2 version);
		maintainer_email: string (server maintainer email);
		hardware: string (host hardware);
		plugins: list (loaded plugins);
		server_time: DateTime (server time);
		processor: string (host processor).
		callable as system.get_properties, web.get_properties.
		"""
		now = time()
		props = self.system_properties.copy()
		# TODO: convert in html_factory, use only floats internally!
		props['server_time'] = DateTime(now)
		props['uptime'] = int(now - props['start_time'])
		props['start_time'] = DateTime(props['start_time'])
		return props
	x_web_get_properties = x_sys_get_properties


######################
# helpers

def is_co2_method(obj):
	"""is_co2_method(obj)
	helper method.
	decide whether or not an object meets the definition of a
	co2-XMLRPC-method, ie. should be registered for XMLRPC access.
	a co2-XMLRPC-method is a class-(?) or instance-method (registry-builtins).
	obj: the object to be checked.
	returns boolean.
	"""
	if inspect.ismethod(obj) and obj.__name__.startswith('x_'):
		return True
	return False

