"""bridge:
co2 plugin that connects pd to co2.
"""

import SocketServer, socket
from time import sleep
from threading import Thread

import plugin
from process import *

class X_Bridge(plugin.Co2Plugin):
	"""X_Bridge
	the bridge class.
	"""
	def __init__(self, var_dict):
		"""__init__(var_dict):
		init plugin.Co2Plugin and then the bridge.
		instantiate and start the receiver, instantiate the sender and send
		pd the init message, instantiate the processor.
		var_dict: dictionary, bridge specific variables (from config file).
		"""
		plugin.Co2Plugin.__init__(self, var_dict)
		from time import time
		self.exit_word = '%s' % str(time())[-8:]
		del time
		receiver_args = (PdReceiver(self.address, PdHandler, self),)
		Thread(target = self.start_receiver, args = receiver_args).start()
		self.sender = PdSender(self.pd_address)
		self.sender.send('sys init %s %d' % self.address)
		self.processor_in = Processor_In()
		self.processor_out = Processor_Out()

	def start_receiver(self, receiver):
		"""start_receiver(receiver):
		start the receiver loop, that handles incoming data from pd.
		receiver: Receiver instance
		"""
		self.running = True
		try:
			# enter receive loop
			while self.running is True:
				receiver.handle_request()
				# pd has closed connection, exit loop and wait for reconnect
				self.debug(self,
									 '--------------END RECEIVE CYCLE-------------------',
									 1)
		except Exception, msg:
			self.debug(self, 'ERROR receiving [%s]' % msg, 3)
			self.__exit__() # etwas drastisch?

	def receive(self, data_dict):
		"""receive(data_dict):
		handle data coming from co2, i.e. preprocess it (see Processor)
		and send it on to pd.
		data_dict: dictionary, whatever data is being relayed.
		"""
		# incoming co2 data
		self.debug(self, 'from co2: %s' % str(data_dict), 0)
		numbers, strings = self.processor_in.process(data_dict['data'])
		self.debug(self,
							 'PROCESSED: nums (%s), strs (%s)' % (numbers, strings),
							 0)
		self.send_pd('co2 num %s' % numbers)
		self.send_pd('co2 sym %s' % strings)

	def receive_pd(self, data):
		"""receive_pd(data):
		handle incoming data from pd. data is wrapped in a dictionary:
		key = 'data', value = data;
		data: string (values [numbers and strings] separated by spaces).
		"""
		self.debug(self, 'from pd: %s' % str(data), 0)
		self.send(self.processor_out.process(data))

	# TODO: separate methods for sys and co2 messages
	def send_pd(self, data):
		"""send_pd(data):
		send data to pd.
		data: string, pd routing prefix plus whitespace separated list of values.
		where prefix is '<channel> [<type>]'. channel is either 'sys' for system calls
		like 'init', 'exit', etc., or 'co2' for relayed co2 data (aka. content) which
		takes an additional type identifier ('num' for numerical values or 'sym' for
		symbols/strings). bridge.pd will route co2 num data to its 'r from_co2_num' or
		and co2 sym data to 'r from_co2_sym' objects.
		"""
		self.debug(self, 'to pd: %s' % str(data), 0)
		self.sender.send(str(data))

	def __exit__(self):
		"""__exit__():
		exit hook, called by the registry on server exit. sends 'sys exit' to
		pd, which is expected to disconnect from our receiver instance, waits
		for pd_grace_period seconds (from config file), and stops the receiver.
		"""
		plugin.Co2Plugin.__exit__(self)
		self.debug(self, 'sending "sys exit" to pd', 1)
		# tell pd we're exiting
		self.send_pd('sys exit %d' % self.pd_grace_period)
		# in case the pd patch wants to say goodbye...
		self.debug(self, 'waiting %d secs...' % self.pd_grace_period, 0)
		sleep(self.pd_grace_period)
		self.debug(self, 'stopping pd receiver', 1)
		# stop listening to pd
		self.running = False
		s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
		s.connect(self.address)
		s.send('%s;\n' % self.exit_word)
		self.debug(self, 'sent exit word to %s:%d' % self.address, 0)
		s.close()
		self.sender.socket.close()

class PdSender:
	"""wrapper around a socket object for sending data to pd
	"""
	def __init__(self, addr):
		"""__init__(addr):
		set up the socket and connect to pd's receiving socket.
		addr: tuple (<host>, <port>), address pd is listening on.
		"""
		self.socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
		self.socket.settimeout(10) # TODO: from config file
		self.socket.connect(addr)

	def send(self, data):
		"""send(data):
		send data to pd.
		data: string, values to be sent to pd.
		"""
		self.socket.send('%s;' % data) # trailing ';' obsolete since pd version 0.36???

class PdHandler(SocketServer.StreamRequestHandler):
	"""handler for messages coming from pd.
	"""
	def handle(self):
		"""handle():
		handle incoming data from pd.
		opens SocketServer.rfile and reads line by line, passes it on to the bridge
		(after stripping pd's terminating ';' and trailing newline) until EOF is
		reached, then starts a new cycle, until the bridge exits.
		"""
		self.debug('connect from %s:%d' % self.client_address, 2)
		for line in self.rfile:
			data = line[:-2] # strip ';' and newline from pd message
##		while True:
##			data = self.rfile.readline()[:-2] # strip ';' and newline from pd message
##			self.rfile.flush()
			self.debug('received %s' % str(data), 0)
			if data == self.server.exit_word and \
						 self.server.host == self.client_address[0]:
				self.debug('received exit secret', 0)
				break
			self.server.receive_pd(data)
		self.debug('stopped listening', 2)

	def debug(self, msg, level):
		"""debug(msg):
		write a debug message to ther server's (co2's) debug file.
		msg: string, log message text.
		"""
		self.server.debug(self, msg, level)

	def __str__(self):
		return self.__class__.__name__

class PdReceiver(SocketServer.ThreadingTCPServer):
	"""server listening to pd.
	"""
	allow_reuse_address = True

	def __init__(self, server_address, RequestHandlerClass, bridge):
		"""__init__(server_address, RequestHandlerClass, bridge):
		first init SocketServer.ThreadingTCPServer, and then some...
		server_address: tuple (<host>, <port>), where we listen to pd,
		RequestHandlerClass: the request handler class (PdHandler);
		bridge: the bridge instance.
		"""
		SocketServer.ThreadingTCPServer.__init__(self,
																						 server_address,
																						 RequestHandlerClass)
		self.receive_pd = bridge.receive_pd
		self.debug = bridge.debug
		self.exit_word = bridge.exit_word
		self.host = socket.gethostbyname(bridge.address[0])

	def server_close(self):
		"""server_close():
		shut down this receiver instance.
		"""
		SocketServer.ThreadingTCPServer.server_close(self)
		self.debug(self, 'receiver closed', 2)

##if __name__ == '__main__':
##	"""for testing"""
##	pass
