"""loader
co2's plugin loader. loads and initializes plugins, and on
server exit calls their exit hooks so they can exit cleanly.
"""

import os, inspect

class Co2LoaderError(Exception):
	"""default loader error, raised whenever something bad happens here.
	"""
	pass

def mod_import(dotted_name):
	"""mod_import(dotted_name):
	import a module with a dotted name.
	dotted_name: the name of the module to import (<package_name>.<module_name>).
	returns the module.
	(taken from the python documentation [i think])
	"""
	mod = __import__(dotted_name)
	components = dotted_name.split('.')
	for comp in components[1:]:
		mod = getattr(mod, comp)
	return mod

class Loader:
	"""Loader:
	the plugin loader class.
	"""
	def __init__(self, vars, plugin_vars):
		"""__init__(vars, plugin_vars):
		init settings and check, raises Co2LoaderError on serious errors.
		vars: dictionary, mandatory keys: base_dir, value='path/to/co2/plugins';
		plugin_list, value = [] or None;
		plugin_vars: dict to pass on to plugins.
		(both arguments from config file.)
		"""
		# funkt so nur, wenn aus dem co2 dir gestartet.
		# im moment(?) OK, weil co2.py dort hin cd't
		self.log = vars['log_method']
		self.debug = vars['debug_method']
		self.plugin_vars = plugin_vars
		self.plugin_list = vars['plugin_list']
		base_dir = vars['base_dir']
		if not os.path.isabs(base_dir):
			raise Co2LoaderError('Loader: base dir must be absolute!')
		if not os.path.isdir(base_dir):
			raise Co2LoaderError('Loader: no such dir [%s]' % base_dir)
		self.base_dir = base_dir
		os.chdir(self.base_dir)
		from sys import path as sys_path
		sys_path.insert(0, self.base_dir)
		del sys_path
		self.debug(self, 'initialized for "%s"' % self.base_dir, 1)

	def load_all(self, exit_hooks):
		"""load_all(exit_hooks):
		load and initialize all plugins from the plugin list.
		exit_hooks: list, instance variable of the registry that owns this loader
		instance. the loader scans vor exit hooks and appends them to the list,
		all members of which will be called on server shutdown.
		"""
		receive_methods = []
		if self.plugin_list is not None:
			if len(self.plugin_list) == 0:
				# plugin_list = [] means load all available
				self.plugin_list = os.listdir(self.base_dir)
			for package_name in os.listdir(self.base_dir):
				if package_name not in self.plugin_list: continue
				package = __import__(package_name)
				self.debug(self, 'loaded package %s' % package_name, 1)
				for module_name in package.__all__:
					dotted_name = '%s.%s' % (package_name, module_name)
					module = mod_import(dotted_name)
					self.debug(self, 'loaded module %s' % dotted_name, 1)
					for object_name, object in inspect.getmembers(module, inspect.isclass):
						if object_name.startswith('X_'):
							plugin_vars = self.plugin_vars
							plugin_vars['dotted_name'] = dotted_name
							plugin_vars['path'] = os.path.join(self.base_dir, package_name)
							plugin_vars['config_filename'] = '%s_config' % module_name
							inst = object(plugin_vars)
							exit_hooks.append(inst.__exit__)
							self.debug(self, 'registered exit_hook [%s]' % inst.__exit__, 0)
							receive_methods.append(inst.receive)
		return receive_methods

	def __str__(self):
		return 'Loader'
	__repr__ = __str__


if __name__ == '__main__':
	"""just for testing"""
	def relay_dummy(data):
		print 'relay:', str(data)
	def debug_dummy(caller, msg):
		print 'debug:', str(caller), msg

	plugin_vars = {'relay_method':relay_dummy,
								 'debug_method':debug_dummy,
								 'log_method':debug_dummy}

	loader_vars = {'base_dir':'/home/lu/devel/co2/co2/plugins',
								 #'plugin_list':['example'],
								 'plugin_list':[],
								 'debug_method':debug_dummy,
								 'log_method':debug_dummy}

	l = Loader(loader_vars, plugin_vars)
	exit_hooks = []
	receive_methods = l.load_all(exit_hooks)
	for recv in receive_methods:
		recv({'data':[1, 2, 'DF'], 'host_id':'sdfasdf@sdva:9093'})
	for ehook in exit_hooks:
		ehook()
