"""satellite
representation of a co2 satellite, used by
relay_registry.Co2RelayRegistry
"""

from threading import Thread
from xmlrpclib import DateTime
from time import time

import client
#import util

class SatelliteError(Exception): pass

class Satellite:
	"""represents a co2 satellite host. has it's own proxy to call
	methods on the satellite, and holds various statistical data as
	instance variables.
	"""
	__props__ = ('host_id', 'nick', 'addr', 'ctime', 'recv_time',
							 'send_time', 'recv_count', 'recv_count', 'ping_current',
							 'ping_min', 'ping_max', 'ping_avg', 'ping_fails',
							 'ping_count', 'plugins')

	def __init__(self, host_id, addr, notify_pass,
							 debug_method, nick='', plugins=[]):
		"""
		@param host_id: the satellite's ID
		@type host_id: string
		@param addr: the address of the satellite
		@type addr: tuple (<host>, <port>)
		@param notify_pass: relay/notify "password"
		@type notify_pass: string
		@param debug_method: the global debug method
		@type debug_method: method object
		@param nick: optional nickname
		@type nick: string
		@param plugins: optional plugin names (loaded by the satellite)
		@type plugins: list of strings
		"""
		self.host_id = host_id
		self.addr = addr
		self.notify_pass = notify_pass
		if nick == '': nick = host_id
		self.nick = nick

		self.proxy = client.Co2Proxy(addr)

		self.ctime = time()

		self.recv_time = self.ctime
		self.recv_count = 0

		self.send_time = self.ctime
		self.send_count = 0

		self.ping_current = -1
		self.ping_min = -1
		self.ping_max = -1
		self.ping_avg = -1
		self.ping_fails = 0
		self.ping_count = 0

		self.debug = debug_method

		self.plugins = plugins

	def set_props(self, props_dict):
		"""set the satellite's properties.

		@param props_dict: property data
		@type props_dict: dictionary, keys must match L{Satellite.__props__}
		"""
		for key in self.__props__:
			try:
				setattr(self, key, props_dict[key])
			except KeyError, msg:
				self.debug(self,
									 'failed to set property "%s" [%s]' % (key, msg),
									 2)

	def get_props(self):
		"""get the satellite's properties.

		@return: properties
		@rtype: dictionary
		"""
		return {'host_id':self.host_id,
						'nick':self.nick,
						'addr':self.addr,
						'plugins':self.plugins,
						'ctime':self.ctime,
						'recv_time':self.recv_time,
						'recv_count':self.recv_count,
						'send_count':self.send_count,
						'send_time':self.send_time,
						'ping_fails':self.ping_fails,
						'ping_current':self.ping_current,
						'ping_min':self.ping_min,
						'ping_max':self.ping_max,
						'ping_avg':self.ping_avg,
						'ping_count':self.ping_count}

	#props = property(get_props, set_props, None, 'satellite properties')

	def get_ctime(self):
		"""get creation time (= registration time).

		@return: timestamp of creation (=registration), seconds since epoche
		@rtype: float
		"""
		return self.ctime

	def touch_recv_time(self):
		"""updates self.recv_time, called whenever a notify-call goes out
		through self.proxy, and increments self.recv_count by one.
		"""
		self.recv_time = time()
		self.recv_count += 1

	def touch_send_time(self):
		"""updates self.send_time, called whenever a relay-call comes in
		from that satellite, and increments self.send_count by one.
		"""
		self.send_time = time()
		self.send_count += 1

	def set_ping_current(self, dt):
		"""set the value of the latest successful ping request, increment
		the ping counter by one, calculate the new ping average, set
		counter for failed pings to 0.

		@param dt: pingtime
		@type dt: float
		"""
		self.ping_count += 1
		self.ping_current = dt
		if self.ping_count == 1:
			self.ping_avg = self.ping_current
			self.ping_min = self.ping_current
			self.ping_max = self.ping_current
		else:
			self.ping_avg += (self.ping_current - self.ping_avg) / self.ping_count
			self.ping_min = min(self.ping_min, self.ping_current)
			self.ping_max = max(self.ping_max, self.ping_current)
		self.ping_fails = 0

	def inc_ping_fails(self):
		"""increment the counter for failed pings by one. called by
		L{pinger.Pinger} when a ping request raises an exception or is not
		returned within L{client.Co2Proxy}'s sockettimeout.

		@return: the updated number of failed pings
		@rtype: int
		"""
		self.ping_count += 1
		self.ping_fails += 1
		return self.ping_fails

	def notify(self, param_dict):
		"""calls the satellites notify method, calls L{Satellite.touch_recv_time}.

		@param param_dict: data to be relayed
		@type param_dict: dictionary
		"""
		if self.ping_fails == 0:
			self.debug(self, 'notify [%s]' % str(param_dict), 0)
			Thread(target = self.notify_thread, args = (param_dict,)).start()
			self.touch_recv_time()
		else:
			self.debug(self,
								 'not notified, failed pings: %s' % self.ping_fails,
								 2)

	def notify_thread(self, param_dict):
		"""notify calls are threaded, so a lame satellite or a bad
		connection don't hurt the relay. called by L{Satellite.notify}.

		@param param_dict: data to be relayed, simply passed on)
		"""
		#print 'Satellite notify thread:', host_id, method_name, data
		param_dict['notify_pass'] = self.notify_pass
		try:
			#self.proxy.proxy.notify(param_dict)
			getattr(self.proxy.proxy, 'notify')(param_dict)
		except Exception, msg:
			#util.trace()
			self.debug(self, 'error in notify thread [%s]' % str(msg), 3)

	def get_status(self):
		"""get the satellites status, as known to the relay.

		@return: various data about this satellite
		@rtype: dictionary,
		format:
		'host_id':string, the satellites ID;
		'addr': tuple, the satellite's address;
		'plugins': list, loaded plugins;
		'ctime': iso8601 DateTime, timestamp of registration (creation);
		'recv_time': iso8601 DateTime, timestamp of most recent receive;
		'recv_count': int, number of messages received;
		'send_time': iso8601 DateTime, timestamp of most recent send;
		'send_count': int, number of messages sent;
		'ping_fails': int, number of currently unanswered pings, reset to
		0 on successful ping;
		'ping_current': float, most recent ping delay;
		'ping_min': float, minimum ping time;
		'ping_max': float, maximum ping time;
		'ping_avg': float, average ping time;
		'ping_count': int, number of successfull pings
		"""
		return {'host_id':self.host_id,
						'nick':self.nick,
						'addr':self.addr,
						'plugins':self.plugins or [],
						'ctime':DateTime(self.ctime),
						'recv_time':DateTime(self.recv_time),
						'recv_count':self.recv_count,
						'send_count':self.send_count,
						'send_time':DateTime(self.send_time),
						'ping_fails':self.ping_fails,
						'ping_current':self.ping_current,
						'ping_min':self.ping_min,
						'ping_max':self.ping_max,
						'ping_avg':self.ping_avg,
						'ping_count':self.ping_count}

	def get_status_raw(self):
		return {'host_id':self.host_id,
						'nick':self.nick,
						'addr':self.addr,
						'plugins':self.plugins or [],
						'ctime':self.ctime,
						'recv_time':self.recv_time,
						'recv_count':self.recv_count,
						'send_count':self.send_count,
						'send_time':self.send_time,
						'ping_fails':self.ping_fails,
						'ping_current':self.ping_current,
						'ping_min':self.ping_min,
						'ping_max':self.ping_max,
						'ping_avg':self.ping_avg,
						'ping_count':self.ping_count}

	def get_dump_data(self):
		props = self.get_status_raw()
		props['notify_pass'] = self.notify_pass
		return props

	def __str__(self):
		return '%s [%s]' % (self.__class__.__name__, self.host_id)
	__repr__ = __str__

if __name__ == '__main__':
	def debug(caller, msg, level=0):
		print '%s: %s' % (caller, msg)
	s = Satellite('ID', ('', 0.815), debug)
	s.set_last_ping(8.9)
	print s.get_status()
