"""relay
the co2 registry with relay functionality
"""

from threading import Timer

import registry, client, util
from satellite import *
from pinger import *

init_ping_delay = 1

class Co2RelayRegistry(registry.Co2Registry):
	"""relay specific registry.
	"""
	def __init__(self, vars):
		"""first calls Co2Registry's __init__, then does some initial stuff
		for its own (load eventual satellite dump file, init and kick off
		pinger and limiter etc..

		@param vars: registry specific variables
		@type vars: dictionary
		"""
		registry.Co2Registry.__init__(self, vars)
		self.system_properties['max_sats'] = vars['max_sats']
		self.satdump_filename = 'saved_sats.xml'
		self.satellites = {}
		self.debug(self, 'looking for dumped satellite data.', 1)
		# check for dumped satellite data
		sat_data = util.read_sat_data(self.vars['co2_home'],
																	self.satdump_filename)
		for sd in sat_data:
			# load dumped sat data
			self.satellites[sd['host_id']] = Satellite(sd['host_id'],
																								 sd['addr'],
																								 sd['notify_pass'],
																								 self.debug)
			self.satellites[sd['host_id']].set_props(sd)
			self.debug(self,
								 'loaded %s' % self.satellites[sd['host_id']],
								 1)
		self.pinger = Pinger(self.satellites, self.debug)
		self.pinger.start()
		self.limiter.init(self._relay_real, self.debug)
		self.limiter.start()

	def _exit_hook(self):
		"""called on server exit. triggers cleaning up.
		"""
		# first do relay specific cleaning up
		if len(self.satellites) > 0:
			try:
				self._serialize_sat_data()
			except Exception, msg:
				self.debug(self,
									 'WARNING: failed to dump satellite data! [%s]' % msg,
									 2)
		else:
			# there are no satellites registered,
			# so remove a dump that might be around
			try:
				if util.remove_file(self.vars['co2_home'],
														self.satdump_filename):
					self.debug(self, 'removed satellite dump file.', 0)
			except Exception, msg:
				self.debug(self,
									 'could not remove file "%s"' % self.satdump_filename,
									 2)
		self.debug(self,
							 'satellite data dumped: %d' % len(self.satellites),
							 1)
		self.pinger.halt()
		# then call registry's exit hook
		registry.Co2Registry._exit_hook(self)
		#self.limiter.halt()

	def _serialize_sat_data(self):
		"""_serialize_sat_data():
		dump an xml file to disc containing all data needed for the relay
		to re-register the satellites automatically on restart, so that
		satellites don't have to register again themselves.
		"""
		if not util.can_marshal:
			self.debug(self,
								 'no marshalling library available. satellite data not saved.',
								 2)
			return
		sat_data = []
		for sat_id in self.satellites.keys():
			#sat_data.append(self.satellites[sat_id].get_props())
			sat_data.append(self.satellites[sat_id].get_dump_data())
		try:
			util.dump_sat_data(sat_data,
												 self.vars['co2_home'],
												 self.satdump_filename)
		except Exception, msg:
			self.debug(self, 'could not dump satellite data [%s]' % msg, 2)

	def _relay_real(self, param_dict):
		"""relay data to all registered satellites (except the sender).
		ie. call their notify method with param_dict.

		@param param_dict: data to be relayed
		@type param_dict: dictionary
		@return: 'OK'
		@rtype: string

		exceptions thrown when making the call to a satellite are logged
		and ignored.
		"""
		self.debug(self, 'relay: %s' % param_dict, 0)
		self._verify_registration(param_dict['host_id'])
		# TODO:
		# record frequency of calls per sat
		# -> pause relaying for sats that are too talkative
		self.satellites[param_dict['host_id']].touch_send_time()
		for sat_id in self.satellites.keys():
			if param_dict['host_id'] != sat_id:
				# relay to (all other) satellites
				param_dict['notify_pass'] = self.satellites[param_dict['host_id']].notify_pass
				try:
					self.satellites[sat_id].notify(param_dict)
				except Exception, msg:
					self.debug(self,
										 'relay: failed to notify %s [%s]' % (sat_id,
																													str(msg)),
										 2)
		return 'OK'

	def _ping(self, satellite):
		# ugly, duplicate code (pinger)...
		from time import time
		t = time()
		try:
			getattr(satellite.proxy.proxy, 'ping')({})
			satellite.set_ping_current((time() - t) * 1000)
			self.debug(self,
								 'init ping %s [%.2fms]' % (satellite.host_id,
																						satellite.ping_current),
								 1)
		except Exception, msg:
			from pinger import max_failed_pings
			#failed_pings = satellite.increment_ping_fails()
			self.debug(self,
								 'failed ping %s (%d/%d)' % (satellite.host_id,
																						 satellite.inc_ping_fails(),
																						 max_failed_pings),
								 2)
			del max_failed_pings
		del time

	########################################
	# xmlrpc methods

	def x_relay_register(self, param_dict):
		"""register a satellite. satellites that want to use relaying must
		call this method first. if the limit self.vars['max_sats'] (set in
		the config file) is reached, an exception is thrown and a Fault is
		returned to the caller.

		@param param_dict: data
		@type param_dict: dictionary:
		'port': the satellites listening port (int),
		'plugins': list of plugin names (strings) loaded by the satellite,
		'nick': nickname (string)
		'notify_pass': sort of a password the relay has to use to be allowed
		to call the registering satellite's notify method, ie. to subsequently
		relay to the satellite.
		@return: the id the satellite has to use for further communication with the relay
		@rtype: string

		callable as: relay.register
		"""
		# we take the "real" IP as provided by the server's handler and the
		# port as supplied by the client to get a reliable return address.
		if len(self.satellites) >= self.vars['max_sats']:
			raise registry.Co2RegistryError('max. # of sats (%d) reached, sorry...' % self.vars['max_sats'])
		client_address = (param_dict['client_address'][0],
											param_dict['port'])
		self.debug(self,
							 'register request for %s:%d' % client_address,
							 1)
		host_id = '%s:%d' % client_address
		self.satellites[host_id] = Satellite(host_id,
																				 client_address,
																				 param_dict['notify_pass'],
																				 self.debug,
																				 param_dict['nick'],
																				 param_dict['plugins'])
		# to have ping props more or less right away
		self.debug(self,
							 'scheduled init ping for %s [%ds]' % (host_id,
																										 init_ping_delay),
							 0)
		Timer(init_ping_delay,
					self._ping,
					[self.satellites[host_id]]).start()
		return host_id

	def x_relay_deregister(self, param_dict):
		"""revoke a satellite's registration.
		should be called by satellites before they shut down. the calling
		client's address must match the supplied host id.

		@param param_dict:
		@type: dictionary ({'host_id':xxx})
		@return: 'OK' on success
		@rtype: string

		@raise KeyError: L{exceptions.KeyError} if no satellite with matching host_id is registered.
		@raise Co2RegistryError: L{registry.Co2RegistryError} if one satellite requests to remove another satellite.

		callable as: relay.deregister
		"""
		host_id = param_dict['host_id']
		self._verify_registration(host_id)
		sat = self.satellites[host_id]
		# a little paranoia: make sure satellites only deregister themselves
		#self._match_id(param_dict['client_address'], sat)
		registry._match_id(param_dict['client_address'], sat)
		del self.satellites[host_id]
		#del self.pinger.satellites[host_id]
		self.debug(self, 'deregistered %s' % host_id, 1)
		return 'OK'

	def x_relay_relay(self, param_dict):
		"""relay data to satellites. this actually only puts the data into
		the L{limiter.Limiter}'s queue.

		@param param_dict: data to be relayed
		@type param_dict: dictionary, must contain a key 'host_id' (value: a registered satellite's id) and a key 'data' (value: whatever data the satellite wants to send)
		@return: 'OK'
		@rtype: string

		@raise Co2RegistryError: L{registry.Co2RegistryError} on failure (returned
		to the caller as L{xmlrpclib.Fault})

		callable as: relay.relay
		"""
		# check senders registration
		self._verify_registration(param_dict['host_id'])
		# match host parts of client-supplied host_id and
		#request's client address
		registry._match_id(param_dict['client_address'],
											 self.satellites[param_dict['host_id']])
		self.limiter.enqueue(param_dict)
		return 'OK'

	def x_relay_get_max_satellites(self, param_dict={}):
		"""get the maximum number of satellites that are allowed to be
		registered simultaniously.

		@return: satellite limit
		@rtype: int

		callable as: relay.get_max_satellites
		"""
		return self.vars['max_sats']

	def x_relay_get_satellites(self, param_dict={}):
		"""get IDs of currently registered satellites.

		@return: host_id's
		@rtype: list of strings

		callable as: relay.get_satellites, co2.get_satellites, web.get_satellites
		"""
		return self.satellites.keys()
	x_co2_get_satellites = x_web_get_satellites = x_relay_get_satellites

	def x_relay_get_num_satellites(self, param_dict={}):
		"""get number of currently registered satellites.

		@return: number of currently registered satellites
		@rtype: int

		callable as: relay.get_num_satellites, co2.get_num_satellites, web.get_num_satellites
		"""
		return len(self.satellites)
	x_co2_get_num_satellites = x_web_get_num_satellites = x_relay_get_num_satellites

	def x_relay_get_satellite_props(self, param_dict):
		"""get status / statistical information from a satellite.

		@param param_dict: data
		@type param_dict: dictionary (key='host_id', value=<host_id>).
		@return: satellite properties
		@rtype: dictionary:
		addr: address (list [<ip>, <port>]),
		host_id: the satellite's id (string),
		nick: the satellite's nickname (string, defaults to host_id),
		ctime: time of registration with the relay (DateTime),
		ping_current: last ping time in msecs, -1 if not pinged (float, xmlrpc: double),
		ping_count: number of pings sent,
		ping_min: shortest ping time (float, xmlrpc: double),
		ping_max: longest ping time (float, xmlrpc: double),
		ping_min: average ping time (float, xmlrpc: double),
		ping_fails: number of failed ping requests by the relay (int);
		send_count: number of relay.relay calls by that satellite (int),
		send_time: most recent message originating from this satellite (DateTime);
		recv_count: number of notify calls to that satellite (int),
		recv_time: most recent message relayed to this satellite (DateTime)

		callable as: relay.get_satellite_props, co2.get_satellite_props, web.get_satellite_props (and co2.get_satellite for backwards compatibility)
		"""
		try:
			return self.satellites[param_dict['host_id']].get_status()
		except KeyError:
			raise registry.Co2RegistryError('no such satellite [%s]' % param_dict['host_id'])
	x_co2_get_satellite_props = x_web_get_satellite_props = x_relay_get_satellite_props
	x_co2_get_satellite = x_relay_get_satellite_props

	def x_relay_get_satellite_props_raw(self, param_dict):
		try:
			return self.satellites[param_dict['host_id']].get_status_raw()
		except KeyError:
			raise registry.Co2RegistryError('no such satellite [%s]' % param_dict['host_id'])

	def x_relay_get_satellite_ctime(self, param_dict):
		"""get a satellites creation (=registration) time.

		@return: satellite's ctime, seconds since epoche
		@rtype: float

		callable as: relay.get_satellite_ctime, co2.get_satellite_ctime, web.get_satellite_ctime
		"""
		return self.satellites[param_dict['host_id']].get_ctime()
	x_co2_get_satellite_ctime = x_web_get_satellite_ctime = x_relay_get_satellite_ctime

	# /xmlrpc methods
	########################################
	# helpers

	def _verify_registration(self, host_id):
		"""verify whether a host id is registered.

		@param host_id: co2 host id
		@type host_id: string

		@raise Co2RegistryError: L{registry.Co2RegistryError} if no satellite with
		the specified id is registered.
		"""
		if not host_id in self.satellites.keys():
			raise registry.Co2RegistryError('%s is not registered.' % host_id)

##	def _match_id(self, address, satellite):
##		"""try to match the host part of an address to the host part of a
##		satellite's host address. minimum strategy to prevent spoofing by
##		matching the satellites address to the client_address of a request.

##		@param address: address to be checked
##		@type address: tuple (host, port)
##		@param satellite: satellite to check against
##		@type satellite: L{relay_registry.Satellite} instance

##		@raise Co2RegistryError: L{registry.Co2RegistryError} if the hosts parts
##		don't match
##		"""
##		# a little paranoia: used to make sure sat_id (provided by sat)
##		# and client_address (passed on by the server's requesthandler) match
##		if satellite.addr[0] != address[0]:
##			raise registry.Co2RegistryError('id mismatch [%s:%d - %s]' % (address,
##																																		satellite.host_id))

##	def _make_host_id(self, addr):
##		"""_make_host_id(addr) -> host id
##		create a unique host id. the supplied address (ip:port) is prefixed by the
##		current timestamp (could be a little more creative...)
##		returns a unique host id.
##		"""
##		return '%s@%s:%d' % (str(int(time())), addr[0], addr[1])

	# /helpers
	########################################

