"""registry:
the core of co2's xmlrpc functionality.
"""

import inspect, os
from sys import path as sys_path
from sys import exit as sys_exit
from threading import Thread
from xmlrpclib import DateTime
from time import time

import client, loader
from limiter import Limiter

class Co2RegistryError(Exception): pass

class Co2Method:
	"""registry-internal representation of a registered (i.e. callable)
	xmlrpc method.
	"""
	def __init__(self, method):
		"""
		@param method: the method as registered by the registry
		@type method: method object
		"""
		self.method = method
		self.method_id = id(method)
		self.doc = method.__doc__ or 'no help available'
		self.signature = format_argspec(inspect.getargspec(self.method))
		#und wenn wir lustig sind, implementieren wir ein get_code ;]

	def __call__(self, param_dict):
		"""call self.method.

		@param param_dict: argument to the method, passed through
		@type param_dict: dictionary
		@return: whatever the called method returns
		"""
		return self.method(param_dict)

	def __str__(self):
		return 'Co2XMLRPCMethod "%s"' % self.method.__name__

def format_argspec(argspec):
	"""create a method signature. will be returned by calls to
	system.methodSignature(<method_name>) and aliases thereof.

	@param argspec: as returned by L{inspect.getargspec}
	@return: formatted argument specs
	@rtype: dictionary, key:value pairs are:
	'args':arguments with no default values (list);
	'args_d':arguments with default values
	(dictionary {'arg':<default value>});
	'args_v':variable arguments (list);
	'args_kw':keyword arguments (list).
	(the latter two are python specific and of no relevance to xmlrpc.)
	"""
	f_args = []
	f_args_w_defaults = {}
	args = argspec[0]
	args.reverse()
	varargs = argspec[1]
	if varargs is None: varargs = []
	varkw = argspec[2]
	if varkw is None: varkw = []
	defaults = argspec[3]
	if defaults is None:
		defaults = []
	else:
		defaults = list(defaults)
		defaults.reverse()
	#print args
	for i in range(len(args)):
		#for i in range(-1, (len(args) + 1) * -1, -1):
		if args[i] == 'self': continue
		try:
			f_args_w_defaults[args[i]] = defaults[i]
		except IndexError:
			f_args.append(args[i])
	return {'args':f_args,
					'args_d':f_args_w_defaults,
					'args_v':varargs,
					'args_kw':varkw}

##class Co2XMLRPCMethod(Co2Method):
##	pass
##class Co2WIMethod(Co2Method):
##	pass

def make_reg_name(name):
	"""create a name under which a method will be registered for access
	through xmlrpc.

	@param name: the name of a co2 method to be registered.
	@type name: string.

	naming conventions:
	prefixes determine whether or not and if under what name methods are
	registered (ie. XMLRPC-callable)
	 methname      XMLRPCname     XMLRPC call
	 x_top_<name>  <name>         proxy.<name>()
	 x_co2_<name>  co2.<name>     proxy.co2.<name>()
	 x_sys_<name>  system.<name>  proxy.system.<name>()
	 (the last one is for userland (xmlrpc.com) compatibility)
	"""
	if name.startswith('x_sys_'): name = 'system.%s' % name[6:]
	elif name.startswith('x_top_'): name = name[6:]
	elif name.startswith('x_co2_'): name = 'co2.%s' % name[6:]
	elif name.startswith('x_relay_'): name = 'relay.%s' % name[8:]
	elif name.startswith('x_web_'): name = 'web.%s' % name[6:]
	# we can assume the rest simply has 'x_' prepended and we
	# remove that for use in plugins where the methodname is
	# appended to package- and module-names, like: pkg.mod.meth
	# don't use in relay or satellite (use x_top_)
	# !!!most of which is deprecated!!!
	else: name = name[2:]
	return name

class Co2Registry:
	"""holds all xmlrpc methods, registers system methods, initializes
	the plugin loader, and handles all incoming calls.
	"""
	def __init__(self, vars):
		"""set variables, load methods, start the loader, register methods.

		@param vars: registry specific variables (from config file)
		@type vars: dictionary
		"""
		self.vars = vars
		self.debug = vars['debug_method']
		self.log = vars['log_method']
		from socket import gethostbyname_ex
		self.vars['relay_hostnames'] = gethostbyname_ex(self.vars['relay_addr'][0])[2]
		self.receivers = [] # receive methods of plugins
		if self.vars['relay_addr']:
			# there's a relay defined, so we need a proxy to talk to it
			# TODO: also use this proxy for relay registry?
			self.relay_proxy = client.Co2Proxy(self.vars['relay_addr'],
																				 self.vars['socket_timeout'])
		self.host_id = 'unregistered@%s:%d' % self.vars['addr']
		self.system_properties = {'co2_version':self.vars['co2_version'],
															'co2_author':self.vars['co2_author'],
															'maintainer':self.vars['maintainer'],
															'maintainer_email':self.vars['maintainer_email'],
															'plugins':self.vars['plugin_list'] or [],
															'start_time':time()}
		plugin_vars = {}
		plugin_vars['relay_method'] = self._relay
		plugin_vars['debug_method'] = vars['debug_method']
		plugin_vars['log_method'] = vars['log_method']
		self.debug(self,
							 'relay_method = %s' % plugin_vars['relay_method'].__name__,
							 0)
		self.methods = {}
		self._load_system_methods()
		self.plugin_exit_hooks = []
		loader_vars = {'base_dir':self.vars['plugin_path'],
									 'plugin_list':self.vars['plugin_list'],
									 'debug_method':self.vars['debug_method'],
									 'log_method':self.vars['log_method']}
		self._load_plugins(loader_vars, plugin_vars)
		self.limiter = Limiter()

	def _relay(self, data_dict):
		"""dummy method to be overridden by subclasses
		(L{satellite_registry.Co2SatelliteRegistry} and
		L{relay_registry.Co2RelayRegistry}.

		@param data_dict: data to be relayed
		@type data_dict: dictionary
		"""
		# desisschiach, pfoa!
		# call_relay sachen jetzt in sat_reg, wo sie eigentlich hingehoeren,
		# aber die sat plugins, die die _relay methode brauchen, werden hier
		# oben initialisiert...
		self.debug(self, 'no relay method available', 2)

	def _load_system_methods(self):
		"""register system (=internal) methods.

		system methods provide meta-information, or utility functions
		(like list all available [callable] methods etc.).
		"""
		for name, method in inspect.getmembers(self, is_co2_method):
			self._register_method(make_reg_name(name), method)
		self.debug(self, '%d builtin methods' % len(self.methods), 1)

	def _load_plugins(self, loader_vars, plugin_vars):
		"""kick off the plugin loader.

		@param loader_vars: loader specific vars
		@type loader_vars: dictionary
		@param plugin_vars: variables to be passed on to plugins
		@type plugin_vars: dictionary
		"""
		l = loader.Loader(loader_vars, plugin_vars)
		self.receivers = l.load_all(self.plugin_exit_hooks)
		self.debug(self, 'receivers:', 0)
		for receiver in self.receivers:
			self.debug('\t', '%s.%s.%s' % (receiver.im_class.__module__,
																		 receiver.im_class.__name__,
																		 receiver.im_func.__name__), 0)
		#self.debug(self, 'receivers:\n%s' % str(self.receivers))

	def _register_method(self, name, method):
		"""register a method to be callable through xmlrpc.

		@param name: the name under which the method will be callable
		@type name: string
		@param method: the method itself
		@type: method object
		"""
		self.methods[name] = Co2Method(method)
		self.debug(self,
							 'registered %s as %s' % (method.__name__, name),
							 1)

	def _exit_hook(self):
		"""called by the server on exit, in turn calls plugins' exit hooks,
		to clean up before exiting.
		"""
		self.debug(self, 'exit_hook called.', 1)
		for hook in self.plugin_exit_hooks:
			self.debug(self, 'calling %s...' % hook, 1)
			hook()
		self.limiter.halt()

##	def _notify_thread(self, recv_method, param_dict):
##		"""call recv_method with param_dict as argument. called by
##		L{Co2Registry.x_top_notify}.

##		@param recv_method: the method to call
##		@type recv_method: method object
##		@param param_dict: argument to recv_method
##		@type param_dict: dictionary
##		"""
##		# TODO: move to satellite_registry!
##		recv_method(param_dict)

##	def _notify(self, recv_method, param_dict):
##		"""call L{Co2Registry._notify_thread} in a thread.

##		@param recv_method: the method to call
##		@type recv_method: method object
##		@param param_dict: argument to recv_method
##		@type param_dict: dictionary
##		"""
##		Thread(target = self._notify_thread, args = (recv_method,
##																								 param_dict)).start()

	def __str__(self):
		return self.__class__.__name__



	####################################################
	# co2 XMLRPC system methods below
	#
	# prefixes determine under what name methods will be
	# registered/accessible, see make_reg_name()

	############
	# top level

	def x_top_ping(self, param_dict={}):
		"""answer a ping request.

		@return: 'pong'
		@rtype: string

		callable as: ping, web.ping
		"""
##		#to test socket timeout
##		from time import sleep
##		sleep(5)
		return 'pong'
	x_web_ping = x_top_ping # registered separately -> alias

	def x_top_test(self, param_dict={}):
		"""just a test method...

		@param param_dict: optional argument
		@type: param_dict: dictionary
		@return: the argument
		@rtype: param_dict

		callable as: test, web.test
		"""
		return 'test OK: %s received %s' % (self.__class__,
																				str(param_dict))
	x_web_test = x_top_test

	def x_top_call_method(self, param_dict):
		"""'anonymized' call method.

		@param param_dict: dictionary with members:
		'method_name':'method_name', plus all arguments that method expects
		(signature returned by x_co2_get_method_help)
		@type param_dict: dictionary
		@return: whatever the called method returns

		callable as: call_method, web.call_method.
		"""
		method_name = param_dict.pop('method_name')
		return self.methods[method_name](param_dict)
	x_web_call_method = x_top_call_method

##	def x_top_notify(self, param_dict):
##		"""distributes incoming calls to all plugins' receive methods.
##		only 

##		@param param_dict: data to be sent to plugins
##		@type param_dict: dictionary
##		@return: 'OK'
##		@rtype: string

##		exceptions thrown by a plugin's receive method are logged and
##		ignored.

##		callable as: notify
##		"""
##		self.debug(self, 'notify [%s]' % param_dict, 0)
##		for recv in self.receivers:
##			try:
##				self._notify(recv, param_dict)
##			except Exception, msg:
##				self.debug(self, 'ERROR:  [%s]' % str(msg), 3)
##				continue
##		return 'OK'

	############
	# co2 level

	def x_co2_get_method_names(self, param_dict={}):
		# TODO: param_dict = prefix -> filter (?)
		"""get the names of available methods.

		@return: method names
		@rtype: list of strings

		callable as: co2.co2_get_method_names, sys.listMethods,
		web.get_method_names
		"""
		return self.methods.keys()
	x_sys_listMethods =	x_web_get_method_names = x_co2_get_method_names
##	print id(x_co2_get_method_names)
##	print id(x_web_get_method_names)
##	print id(x_sys_listMethods)

	def x_co2_get_method_help(self, param_dict):
		"""get help (the __doc__ string) for a specific method.

		@param param_dict: data
		@type param_dict: dictionary,
		mandatory key: 'method_name', value: name of a method
		(from list returned by x_co2_get_method_names)
		@return: a brief description of what the method does
		@rtype: string

		callable as: co2.get_method_help, sys.methodHelp, web.get_method_help
		"""
		return self.methods[param_dict['method_name']].doc
	x_sys_methodHelp = x_co2_get_method_help
	x_web_get_method_help = x_co2_get_method_help

	def x_co2_get_method_signature(self, param_dict):
		"""get a discription of how to use a specific method.

		@param param_dict: data
		@type param_dict: dictionary,
		mandatory key: 'method_name', value: (callable) name of a method
		@return: method signature
		@rtype: dictionary:
		'args': list of argument names (strings),
		'args_d': dictionary (key=argument name, value=argument value),
		'args_v': list of variable arguments (unused in xmlrpc),
		'args_kw': list of keyword arguments (unused in xmlrpc)

		callable as: co2.get_method_signature, sys.methodSignature, web.get_method_signature
		"""
		return self.methods[param_dict['method_name']].signature
	x_sys_methodSignature = x_co2_get_method_signature
	x_web_get_method_signature = x_co2_get_method_signature

	############
	# sys level
	def x_web_get_properties(self, param_dict={}):
		"""get system properties.

		@return: system properties
		@rtype: dictionary [key: value-type (description)]:
		co2_author: string (server author);
		kernel: string (host kernel name);
		uptime: int (server uptime in seconds);
		operating_system: string (host operating system);
		maintainer: string (server maintainer);
		start_time: DateTime (server start time);
		sw_platform: string (co2 software platform);
		hw_platform: string (host hardware platform);
		co2_version: string (co2 version);
		maintainer_email: string (server maintainer email);
		hardware: string (host hardware);
		plugins: list of strings (loaded plugins);
		server_time: DateTime (server time);
		processor: string (host processor)

		callable as: system.get_properties, web.get_properties
		"""
		now = time()
		props = self.system_properties.copy()
		# TODO: convert in html_factory, use only floats internally!
		props['server_time'] = DateTime(now)
		props['uptime'] = int(now - props['start_time'])
		props['start_time'] = DateTime(props['start_time'])
		return props
	x_sys_get_properties = x_web_get_properties


######################
# helpers

def is_co2_method(obj):
	"""helper method.
	decide whether or not an object meets the definition of a
	co2-XMLRPC-method, ie. should be registered for XMLRPC access.
	a co2-XMLRPC-method is a class-(?) or instance-method whose name
	begins with 'x_' and is built into the registry.

	@param obj: the object to be checked
	@type obj: object
	@rtype: boolean
	"""
	if inspect.ismethod(obj) and obj.__name__.startswith('x_'):
		return True
	return False

def _match_id(address, satellite):
	"""try to match the host part of an address to the host part of a
	satellite's host address. minimum strategy to prevent spoofing by
	matching the satellites address to the client_address of a request.

	@param address: address to be checked
	@type address: tuple (host, port)
	@param satellite: satellite to check against
	@type satellite: L{relay_registry.Satellite} instance

	@raise Co2RegistryError: L{registry.Co2RegistryError} if the hosts parts
	don't match
	"""
	# a little paranoia: used to make sure sat_id (provided by sat)
	# and client_address (passed on by the server's requesthandler) match
	if satellite.addr[0] != address[0]:
		raise Co2RegistryError('id mismatch [%s - %s]' % (address[0],
																											satellite.host_id))
