# example.py
#
# a _very_ basic co2 plugin

# first, we have to import the module 'plugin' which defines the
# crucial Co2Plugin class
# you will have to copy plugins/example/plugin.py to your package
# directory so you can import it (a symlink also works).
# or you extend sys.path to include plugins/example

import plugin


# we subclass Co2Plugin...
# (the name starts with 'X_' so the loader loads it)

class X_Plugin1(plugin.Co2Plugin):
	# ...and override its receive method
	def receive(self, data_dict):
		# all data relayed to the satellite will be passed here.

		# we simply write it out to the server's debug log.
		# (set the server's debug level to 0 or change the last arg to a higher
		# value so that the messages are logged.)
		self.debug(self, data_dict, 0)

		# and send it straight back out
		# which basically is a bad idea, because if all hosts (or, actually, only
		# two) do that, we end up with a feedback loop...
		# so, don't run this!
		self.send(data_dict)


# we make another subclass
# yes, that's right, any number of Co2Plugin subclasses are allowed in one
# plugin. and they all can receive and send independently (for what that is
# worth..., but, hey, it comes at no extra cost! but in real life, you should
# probably make a seperate package for seperate tasks.)

class X_Plugin2(plugin.Co2Plugin):

	# we need some initialisation, so we override plugin.Co2Plugin's __init__
	def __init__(self, var_dict):
		# but call the superclass's init (important!)
		plugin.Co2Plugin.__init__(self, var_dict)

		# and do some additional init stuff:
		# instantiate some class that does work for us
		# (here it becomes clear why we insist on plugins to be branded with the
		# 'X_' prefix. so that we can have any number of classes here that we might
		# need but are not the plugin-loaders business)
		self.helper = Helper()

	# again, we override receive
	def receive(self, data_dict):
		# we get the actual data out of the data dictionary (see the protocol specs)
		# and let the helper process it
		# n.b.: if the processing takes a long time then it should by any means be
		# done in a thread, in order to not keep the relay waiting for an answer
		# too long. (TODO: example)
		self.debug(self, 'received: %s' % str(data_dict), 0)
		processed_data = self.helper.process(data_dict['data'])
		if processed_data is not None:
			# we only send data back if the helper returned some
			self.send({'data':processed_data})

# our helper class that does some work
class Helper:
	def __init__(self):
		# it's so stupid, we don't even have to init it.
		# but, of course, a real helper would probably do a lot of stuff here
		# or some place else (remember, you can have any number of classes,
		# methods, etc. and also import other modules which can also reside
		# in this package's directory whithout interfering with the plugin
		# architecture as long as they are not listed in the package's
		# __all__ variable.
		pass
	def process(self, data):
		# we only process strings that don't contain caps
		if type(data) == type('') and data.islower():
			# (which simply are uppercased)
			return data.upper()
		# ...and integers...
		elif type(data) == type(0):
			return data * 12.3
		# ...and floats...
		elif type(data) == type(0.0):
			# (which we convert to strings and prepend with a lowercase character)
			return 'f%s' % str(data)
		else: return None

# TODO:
# - write a plugin that sends without being triggered by receive!
# - threaded plugin example
