"""pinger
periodically ping registered satellites.
"""

from threading import Thread
from time import time, sleep

# TODO: these from config file:
max_failed_pings = 5 # after which we delete the satellite
tick = 1 # Pinger frequency in secs
ping_iv = 60 # ping cycle in ticks

class Pinger(Thread):
	"""pings all registered satellites and kicks them when they're
	unreachable for too long.
	"""
	def __init__(self, satellites, debug_method):
		"""
		@param satellites: registered satellites.
		@type satellites: dictionary, key=host_id, value=L{satellite.Satellite}
		instance.
		@param debug_method: the global debug method.
		@type debug_method: method object (the global debug method).
		"""
		Thread.__init__(self)
		self.satellites = satellites
		self.debug = debug_method

	def run(self):
		"""start the ping cycle.
		the thread will ping all registered satellites (starting a separate
		thread for each, sleep for one tick (default 1s), then check if the
		halt method has been called, if so, exit the loop, otherwise sleep
		another tick etc.
		"""
		self.running = True
		count = 0
		while self.running:
			if (count % ping_iv) == 0:
				self.debug(self, 'entering ping cycle', 0)
				for host_id in self.satellites.keys():
					Thread(target = self.ping,
								 args = (self.satellites[host_id],)).start()
			count += 1
		 	sleep(tick) # TODO: rather use abs time?
		self.debug(self, 'exiting...', 1)

	def ping(self, satellite):
		"""call a satellite proxy's ping method, calculate the delay, store
		it in the satellite's local representation and log the delay to the
		server's debugfile.
		@param satellite: the satellite to ping.
		@type satellite: a L{satellite.Satellite} instance.
		"""
		t = time()
		try:
			getattr(satellite.proxy.proxy, 'ping')({})
			satellite.set_ping_current((time() - t) * 1000)
			self.debug(self, '%s: time = %.2fms]' % (satellite.host_id,
																							 satellite.ping_current),
								 1)
		except Exception, msg:
			#import util
			#util.trace()
			failed_pings = satellite.inc_ping_fails()
			self.debug(self,
								 '%s: failed [%d/%d]' % (satellite.host_id,
																				 failed_pings,
																				 max_failed_pings),
								 2)
			if failed_pings > max_failed_pings:
				del self.satellites[satellite.host_id]
				self.debug(self,
									 'kicked %s [failed %d]' % (satellite.host_id,
																							failed_pings),
									 2)

	def halt(self):
		"""
		exit the ping cycle on next tick.
		called on L{relay_registry.Co2RelayRegistry} exit.
		"""
		self.running = False
		self.debug(self, 'HALT', 0)

	def __str__(self):
		return self.__class__.__name__
	__repr__ = __str__

