from threading import Thread
from time import sleep

class Limiter(Thread):
	"""FIFO like class for throttling co2 traffic by limiting the
	throughput of calls. used in both, satellite and relay.
	"""
	# TODO: these from config file
	max_q_len = 50
	iv = 0.1
	# rediculously low settings for testing
	#max_q_len = 2
	#iv = 1

	def init(self, method, debug_method):
		"""do registry-specific initilization.

		@param method: is called with the next item in the queue as argument
		(see L{Limiter.pop})
		@type method: method object
		@param debug_method: the global debug method
		@type debug_method: method object
		"""
		self.queue = []
		self.method = method
		self.debug = debug_method
		self.running = False

	def enqueue(self, data):
		"""append a data item to the queue.

		@param data: the data to be appended.
		@type data: any.
		"""
		if len(self.queue) < self.max_q_len:
			self.queue.append(data)
			#self.debug(self, 'enqueue: q = %s' % str(self.queue))
		else:
			self.debug(self,
								 'WARNING: queue full. %s dropped!' % str(data),
								 2)
			#self.debug(self, 'q = %s' % str(self.queue))

	def run(self):
		"""start the limiter cycle.

		the thread checks the queue every L{Limiter.iv} seconds for new
		entries. if more than 0 are found the oldest (index 0) is
		L{popped<Limiter.pop>}
		"""
		self.running = True
		self.debug(self, 'start [max_q_len=%d, iv=%.2f]' % (self.max_q_len,
																												self.iv), 1)
		while self.running:
			if len(self.queue) > 0:
				self.pop()
			if not self.running: break
			sleep(self.iv)
		self.debug(self, 'exiting...', 1)

	def halt(self):
		"""activate the exit condition for the main loop. causes the thread
		to exit after at most L{Limiter.iv} seconds.
		"""
		self.debug(self, 'HALT', 0)
		self.running = False

	def pop(self):
		"""pop the oldest item from the queue and call self.method with i
		as argument.
		"""
		#self.debug(self, 'bfore pop: q = %s' % str(self.queue))
		data = self.queue.pop(0)
		self.method(data)
		if len(self.queue) == 0:
			self.debug(self, 'queue empty', 0)

	def __str__(self):
		return self.__class__.__name__

if __name__ == '__main__':
	"""for testing."""
	def debug(caller, msg, level=0):
		print '%s: %s' % (caller, msg)
	def meth(data):
		debug('meth', str(data))
	l = Limiter()
	l.init(meth, debug)
	for key in range(0, 10):
		l.enqueue({str(key):key})
	l.start()
	sleep(2)
	for key in range(10, 15):
		l.enqueue({str(key):key})
	sleep(.2)
	for key in range(10, 15):
		l.enqueue({str(key):key})
	sleep(1)
	l.halt()
