"""co2
wrapper around a co2 server."""

import sys, os, getopt
from time import strftime

import server

__version__ = '0.5.11'
__author__ = 'lu'

default_debug_level = 2
debug_level = default_debug_level

def usage(msg=None):
	"""print out useage info.

	@param msg: optional additional message
	@type msg: string
	"""
	if msg: print 'Co2: %s' % msg
	print """usage: python %s [-c config-file]
options:
  -c <filename>: config-file (default: config)
arguments: none
""" % sys.argv[0]

def debug(caller, msg, level=0):
	"""the global debug method passed on to all components that might
	want to debug in the central place (stdout or file, depending on
	the server config.)

	@param caller: prefix for the entry
	@type caller: string or object with a __repr__ method
	@param msg: debug message text
	@type msg: string
	"""
	if level >= debug_level:
		print '[%s] %s: %s' % (make_log_timestamp(), caller, msg)

def make_log_timestamp():
	return strftime('%Y.%m.%d %H:%M:%S')

class Co2:
	"""the actual wrapper object.
	"""
	def __init__(self, s_vars, r_vars):
		"""instantiate a server (satellite or registry).
		exits in case the initialization of L{server.Co2Server} fails.
		handles keyboard interrupts and signals (-> exit cleanly).
		@param s_vars: server specific variables
		@type s_vars: dictionary
		@param r_vars: registry specific variables
		@type r_vars: dictionary
		"""
		self.s = server.Co2Server(s_vars)
		try:
			r = registry_class(r_vars)
		except Exception, msg:
			debug(self, 'ERROR: failed to initialize registry [%s]' % msg, 3)
			import util
			util.trace(True)
			sys.exit(1)
		self.s.register_instance(r)
		if hasattr(r,'_exit_hook'): self.s.register_exit_hook(r._exit_hook)
		import signal
		signal.signal(signal.SIGHUP, self.sig_handler)
		signal.signal(signal.SIGTERM, self.sig_handler)
		del signal
		try:
			self.s.listen()
		except KeyboardInterrupt:
			debug(self, 'caught KeyboardInterrupt. exiting...', 3)
			self.shut_down()

	def sig_handler(self, signum, frame):
		"""sig_handler(signum, frame):
		handle a signal. at the moment SIGHUP and SIGTERM are handled,
		i.e. the server tries to exit cleanly.
		"""
		debug(self, 'caught signal %d, exiting...' % signum, 3)
		#self.shut_down() ## warum geht das nicht wirklich???
		raise KeyboardInterrupt ## und das schon???

	def shut_down(self):
		"""shut_down():
		shut down the server cleanly.
		"""
		self.s.clean_up()
		self.s.server_close()
		debug(self, 'exiting', 2)

	def __str__(self):
		return 'Co2Wrapper'
	__repr__ = __str__


def setUID(uid):
	"""set the UID of this process. on failure, prints a warning and
	tries to continue without setting the UID.

	@param uid: numerical user ID
	@type uid: int
	"""
	try:
		os.setuid(uid)
	except Exception, msg:
		debug('Co2', 'WARNING: could not set UID [%s]' % msg, 2)
	debug('Co2', 'INFO: running as UID "%d"' % os.getuid(), 1)

def daemonize(log_file):
	"""fork into the background and redirect all output to a file.

	@param log_file: filename (absolute path) of the logfile to use
	@type log_file: string
	"""
	# redirect stdout and stderr
	try:
		sys.stdout = sys.stderr = file(log_file, 'a', 0)
	except IOError, msg:
		debug('Co2', 'ERROR: could not open logfile [%s]' % msg, 3)
		sys.exit(1)

	# double-fork to detach the process properly
	# fork 1
	pid = os.fork()
	if pid != 0:
		sys.exit(0) # parent exits
	debug('Co2', 'fork 1 [PID=%d]' % os.getpid(), 0)
	os.setsid()

	# fork 2
	pid = os.fork()
	if pid != 0:
		sys.exit(0) # parent exits
	# child, update pid (=0 after fork())
	pid = os.getpid()
	debug('Co2', 'fork 2 [PID=%d]' % pid, 0)

	os.chdir(config.co2_home)

	# write PID to file
	try:
		pid_file = file(global_vars['pid_file'], 'w')
		pid_file.write(str(pid))
		pid_file.close()
	except: debug('Co2',
								'WARNING: failed to open %s' % global_vars['pid_file'],
								2)

	os.umask(0)

	devnull = os.open('/dev/null', 0)
	os.dup2(devnull, 0)
	os.dup2(devnull, 1)
	os.dup2(devnull, 2)

if __name__ == '__main__':
	"""start a co2 server"""
	try:
		options, arguments = getopt.getopt(sys.argv[1:],
																			 'c:',
																			 ['config'])
	except getopt.GetoptError, msg:
		usage(msg)
		sys.exit(1)

	config_file = 'config' # default config file
	cmdline_b = False

	for o, a in options:
		if o == '-c':
			try: a = a[:a.rindex('.py')] # strip evtl. python suffixes
			except: pass
			config_file = a

	# load configuration
	try: config = __import__(config_file)
	except Exception, msg:
		debug('Co2', 'ERROR: failed to load config "%s" [%s]' % (config_file,
																														 msg), 3)
		sys.exit(1)

	global_vars = config.global_vars
	debug_level = global_vars['debug_level']

	registry_vars = config.registry_vars
	if len(arguments) > 0:
		registry_vars['plugin_list'] = arguments
	registry_vars['co2_version'] = __version__
	registry_vars['co2_author'] = __author__

	server_vars = config.server_vars
	# TODO: separate these:
	server_vars['debug_method'] = registry_vars['debug_method'] = debug
	server_vars['log_method'] = registry_vars['log_method'] = debug

	if config.registry == 'satellite':
		import satellite_registry
		registry_class = satellite_registry.Co2SatelliteRegistry
		del satellite_registry
	elif config.registry == 'relay':
		import relay_registry
		if registry_vars.has_key('init_ping_delay'):
			relay_registry.init_ping_delay = registry_vars.pop('init_ping_delay')
		registry_class = relay_registry.Co2RelayRegistry
		del relay_registry
	else:
		usage('no or bogus registry "%s". available: %s' % (str(config.registry),
																												str(config.registries)))
		sys.exit(1)

	#if config.background or cmdline_b:
	if config.background:
		daemonize(global_vars['log_file'])

	if config.uid:
		# change uid
		setUID(config.uid)

	debug('Co2', 'global_vars: %s' %  str(global_vars), 0)
	debug('Co2', 'server_vars: %s' %  str(server_vars), 0)
	debug('Co2', 'registry_vars: %s' % str(registry_vars), 0)

	# finally, start the server
	Co2(server_vars, registry_vars)

	# clean up after the server has exited
	try:
		debug('Co2',
					'deleting pid file "%s"' % os.path.join(global_vars['co2_home'],
																									global_vars['pid_file']), 0)
		os.remove(os.path.join(global_vars['co2_home'], global_vars['pid_file']))
		debug('Co2', 'deleted pid file', 0)
	finally:
		# necessary???
		sys.stdout.close()
		sys.stderr.close()
	print 'Co2', 'ahoi'
